#include "stdafx.h"
#include "version.h"
#include "RwExp.h"

#include <archive\rpanim.h>
#include <rpskin.h>
#include <rpusrdat.h>

#include "max.h"
#include "RwCommon.h"
#include "DffExp.h"
#include "common.h"
#include "utilities.h"
#include "matrixmangle.h"
#include "hanim.h"

#include "ikctrl.h"

#include "WarningsDlg.h"

#include "RwExpError.h"

#include "keysample.h"

//character studio stuff should be migrated outta here and into characterstudio.cpp
#include "characterstudio.h"

//selsetskin stuff should be migrated outta here and into selsetskin.cpp
#include "selsetskin.h"

//max native skin stuff should be migrated outta here and into maxnativeskin.cpp
#include "maxnativeskin.h"

#include "Particles.h"
#include "bezier.h"

#include "float.h"

#include <assert.h>

#define CSEXTRABONETAGINDEXOFFSET 2000

/*********************************************************

    Code to check if material colors need modulating

  ********************************************************/

RwBool GeometryHasColoredMaterials(RpGeometry *geometry)
{
    RwBool colored = FALSE;

    RpGeometryForAllMaterials(geometry, checkMaterialColors, &colored);

    return colored;
}

RwBool GeometryHasMaterialEffects(RpGeometry *geometry)
{
    RwBool effects = FALSE;

    RpGeometryForAllMaterials(geometry, checkMaterialEffects, &effects);

    return effects;
}

class ClumpWrapper
{
public:
    ClumpWrapper()
    {
        m_obj = RpClumpCreate();
    }

    ~ClumpWrapper()
    {
        if (!m_obj)
        {
            return;
        }

        //if this got called while an exception was thrown...
        try
        {
            RpClumpDestroy( m_obj );
        }
        //...we'd better catch anything Destroy might throw or we'll crash instantly
        //(e.g. RwDebugAssertHandler is overloaded to throw an exception)
        catch (...)
        {

        }
    }

    //access to the object
    RpClump * GetClump() { return m_obj; }

private:
    RpClump *m_obj;
};

class NegativeScaleChecker
{
public:
    NegativeScaleChecker(TimeValue when, WarningList &warningList )
        : m_when(when), m_warningList(warningList) {}

    void operator()(INode *node)
    {
        NegativeScaleCheck( node, m_when, m_warningList );
    }

private:
    TimeValue m_when;
    WarningList &m_warningList;
};

RwExpError
DFFExport::TraverseScene(const TCHAR *filename, Interface *gi )
{
    RwStream *stream;
    int nNumChildren, nChildNum;

    CSBoneIndexNodePointers = NULL;
    CSBoneIndexTags = NULL;
    /* We start the extra bone tags at an offset since users tend to tag bones
       starting at zero and we don't want to clash. In the future we'll check the
       full hierarchy to ensure tags don't clash */
    CSExtraBoneTagIndex = CSEXTRABONETAGINDEXOFFSET;
    CSNumBones = 0;
    CSBipedName = NULL;
    TCHAR sFileName[MAX_PATH];

    rootNode = gi->GetRootNode();

    if (!rootNode)
    {
        return RwExpError("Couldn't export - no root node");
    }

    /* Get the starting time value of the animation */
    m_tvStart = gi->GetAnimRange().Start();
    m_tvEnd = gi->GetAnimRange().End();

    /* Calculate the number of frames in the sub anim */
    m_nNumFrames = ((m_tvEnd - m_tvStart) / (GetTicksPerFrame())) + 1;

    /* Recalculate the scale factor if we're using "scale to" */
    if (m_nScaleType == 1) /* scale to */
    {
        m_nScaleFactor = ScaleTo_ScaleFactor(rootNode);
    }

    /* Build a scale matrix */
    m_scaleMatrix.IdentityMatrix();
    m_scaleMatrix.Scale(Point3(m_nScaleFactor,m_nScaleFactor,m_nScaleFactor), TRUE);

    //issue warnings if some scene nodes have negative scaling
    NegativeScaleChecker nsc( m_tvStart, m_WarningsList );
    ForEachNodeInHierarchy(rootNode, nsc );

    SKAKeyFrameList.next = NULL;
    AnimKeyFrameList.next = NULL;
    skinMatrixWeights = NULL;
    skinVertexIndices = NULL;
    skinInverseMatrices = NULL;
    skinFlags = NULL;

    /* convert the selected clumps */

    /* The selected and child nodess are stored as member data. */
    RwExpError msg = ExportSelectFirstObjectNode(gi);
    if(msg.Explanation() != "")
    {
        return msg;
    }

    do
    {
        InitialiseCustomNodeData(m_topOfSelectedHierarchy);   //DNGN

        //search scene for skin & bones nodes that the user selected.
        INode * maxSkinNode = 0;
        if (m_maxNativeSkin)
        {
            RwExpError msg = FindMaxSkinAndBonesNodes(rootNode, m_topOfSelectedHierarchy,
                maxSkinNode, m_topOfSelectedHierarchy );

            if (msg.Explanation() != "")
            {
                return msg;
            }
        }

        //search scene for physique & biped nodes that the user selected.
        BonesContainer influenceBones;
        SkinsContainer physiquesToExport;
        if (m_CSExport)
        {
            RwExpError msg = FindPhysiqueAndBipedNodes( gi, &m_WarningsList,
                rootNode, m_topOfSelectedHierarchy,
                physiquesToExport, influenceBones );

            if (msg.Explanation() != "")
            {
                return msg;
            }

            //start from the hierarchy containing any bone.
            //(won't work for floating bones, but we don't handle those yet).
            if ( !influenceBones.empty() )
            {
                m_topOfSelectedHierarchy = FindTopOfHierarchy( *(influenceBones.begin()) );
            }

            //name of first node in subtree with a biped controller (safe?!)
            CSBipedName = FindBipedNameInHierarchy(m_topOfSelectedHierarchy);
        }

        //make a clump to hold exported contents of this heirarchy
        ClumpWrapper clump;
        if (!clump.GetClump())
        {
            throw RwExpError("Failed to create a clump");
        }
        else
        {
            /* Log the creation of a new clump. */
            m_WarningsList.newClump(rootNode->GetName());
        }

        RwFrame *frame = RwFrameCreate();
        if (!frame)
        {
            throw RwExpError("Failed to create a frame");
        }
        else
        {
            /* Log the creation of a new frame. */
            m_WarningsList.newFrame(rootNode->GetName());
        }

        RpClumpSetFrame(clump.GetClump(), frame);

        //recursively mangle child nodes into atomics within this clump!
        ConvertNode(m_topOfSelectedHierarchy, clump.GetClump(), frame, TRUE, influenceBones);

        /* Knock out the push on the root */
        if (skinFlags)
        {
            skinFlags[0] &= ~rpSKINPUSHPARENTMATRIX;
        }
        if (m_exportHAnim)
        {
            //For HAnim we now create an RpHAnimHierarchy and attach it to the root frame
            RpHAnimHierarchy *pHierarchy;
            RwInt32 HAnimFlags = 0;
            if (m_HAnimLocalSpaceMatrices)
            {
                HAnimFlags |= rpHANIMHIERARCHYLOCALSPACEMATRICES;
            }
            if (m_HAnimNoMatrices)
            {
                HAnimFlags |= rpHANIMHIERARCHYNOMATRICES;
            }

            pHierarchy = RpHAnimHierarchyCreate(CSNumBones, skinFlags, CSBoneIndexTags, (RpHAnimHierarchyFlag)HAnimFlags, rpHANIMSTDKEYFRAMESIZE);
            RpHAnimSetHierarchy(RpClumpGetFrame(clump.GetClump()), pHierarchy);
        }

        //If we are exporting animation data where the rootnode is animated
        //or if we're doing some skinned animation
        //insert a placeholder frame (identity transform) at the root of the clump hierarchy
        //this will allow world offsets to be applied to the animated heirarchy
        if (    (m_anim && m_topOfSelectedHierarchy->IsAnimated())
            ||  m_skinning || m_exportHAnim )
        {
            RwFrame *root = RwFrameCreate();
            if (!root)
            {
                throw RwExpError("Failed to create a frame");
            }
            else
            {
                /* Log the creation of a new frame. */
                m_WarningsList.newFrame(rootNode->GetName());
            }

            RwMatrixSetIdentity(RwFrameGetMatrix(root));
            RwFrameAddChild(root, RpClumpGetFrame(clump.GetClump()));
            RpClumpSetFrame(clump.GetClump(), root);
            RwFrameUpdateObjects(root);

            InitialiseCustomFrameData(m_topOfSelectedHierarchy, root);
        }

        if (m_seqAnim || m_skinning)
        {
            AddAnim(m_topOfSelectedHierarchy, clump.GetClump(), m_topOfSelectedHierarchy->GetName());
        }

        if (m_CSExport || m_linkBySelSets || m_maxNativeSkin)
        {
            //We'd really prefer it if these didn't have to be mutually exclusive,
            //because you can combine bipeds & max native skinning in MAX, it just won't export yet.
            if (m_maxNativeSkin)
            {
                ProcessMaxSkinNodesInHierarchy(maxSkinNode, clump.GetClump());
            }
            else if (m_CSExport)
            {
                ProcessPhysiqueNodes(this, physiquesToExport, rootNode, clump.GetClump());
            }
            else if (m_linkBySelSets)
            {
                ProcessSelSetSkinNodesInHierarchy(m_topOfSelectedHierarchy, m_topOfSelectedHierarchy, clump.GetClump());
            }

            if (CSBipedName)
            {
                RwFree(CSBipedName);
                CSBipedName = NULL;
            }

            InitialiseCustomFrameData(m_topOfSelectedHierarchy, RpClumpGetFrame(clump.GetClump()));
        }

        InitialiseCustomFrameData(m_topOfSelectedHierarchy, frame);

        if (m_skinning && m_skinSKA)
        {
            RpSkinAnim *skinAnim = GenerateSkinAnimFromAnimKeyFrameList(&SKAKeyFrameList, CSNumBones);

            /* write out the first .ska based on the name of the export */
            RwChar *extender = strrchr(filename, '.');
            *extender = 0x0;
            RwChar *animName = (char *)RwMalloc(sizeof(char) * (strlen(filename) + 5));
            strcpy(animName, filename);
            strcat(animName, ".ska");
            *extender = '.';
            RpSkinAnimWrite(skinAnim, animName);
            RpSkinAnimDestroy(skinAnim);
            /* Log the creation of the ska file. */
            m_WarningsList.writtenFile(animName);
        }
        else
        {
            DestroyKeyFrameList(&SKAKeyFrameList);
        }
        if (m_exportHAnim)
        {
            RpHAnimAnimation *HAnim = GenerateHAnimationFromKeyFrameList(&AnimKeyFrameList);

            /* write out the first .ska based on the name of the export */
            RwChar *extender = strrchr(filename, '.');
            *extender = 0x0;
            RwChar *animName = (char *)RwMalloc(sizeof(char) * (strlen(filename) + 5));
            strcpy(animName, filename);
            strcat(animName, ".anm");
            *extender = '.';
            RpHAnimAnimationWrite(HAnim, animName);
            RpHAnimAnimationDestroy(HAnim);
            /* Log the creation of the ska file. */
            m_WarningsList.writtenFile(animName);
        }
        else
        {
            DestroyKeyFrameList(&AnimKeyFrameList);
        }

        if (m_seqAnim || m_linkBySelSets)
        {
            nNumChildren = rootNode->NumberOfChildren();
            for (nChildNum = 0; nChildNum < nNumChildren; nChildNum++)
            {
                INode *childNode = rootNode->GetChildNode(nChildNum);

                if (childNode != m_topOfSelectedHierarchy)
                {
                    AddAnim(childNode, clump.GetClump(), childNode->GetName());
                    if (m_skinning)
                    {
                        RpSkinAnim *skinAnim = GenerateSkinAnimFromAnimKeyFrameList(&SKAKeyFrameList, CSNumBones);

                        /* write another .ska based on the name hiearchy */
                        RwChar *animName = (char *)RwMalloc(sizeof(char) * (strlen(childNode->GetName()) + 5));
                        strcpy(animName, childNode->GetName());
                        strcat(animName, ".ska");
                        RpSkinAnimWrite(skinAnim, animName);
                        RpSkinAnimDestroy(skinAnim);
                        /* Log the creation of the ska file. */
                        m_WarningsList.writtenFile(animName);
                    }
                    else
                    {
                        DestroyKeyFrameList(&SKAKeyFrameList);
                    }
                }
            }
        }

        if (m_skinning)
        {
            {
                int i;
                for (i=0; i<CSNumBones; i++)
                {
                    RWEXPMESSAGE(("%s(%d): %d PUSH=%d, POP=%d\n",
                                  __FILE__, __LINE__,
                                  i,
                                  skinFlags[i] & rpSKINPUSHPARENTMATRIX,
                                  skinFlags[i] & rpSKINPOPPARENTMATRIX));
                }
            }
            RwFree(skinInverseMatrices);
            RwFree(CSBoneIndexNodePointers);
            CSBoneIndexNodePointers = NULL;
        }

        /* add all the sequences to the clump */
        if (m_anim || m_seqAnim)
        {
            _rpAnimClumpForAllFramesAddSequences(clump.GetClump());
        }

        SetCustomNodeData(m_topOfSelectedHierarchy);

        /* save out the clump */
        if (RpClumpGetNumAtomics(clump.GetClump()) == 0)
        {
            char temp[256];
            sprintf(temp,"RenderWare %s Dff Exporter",VersionNumberText);
            MessageBox(hGMaxHWnd, "Warning: Clump will be exported but contains no atomics",
                temp, MB_OK );
        }

        /* Customise the file name. */
        strcpy(sFileName, filename);
        CustomiseFileName(sFileName);

        if ((stream = RwStreamOpen(rwSTREAMFILENAME, rwSTREAMWRITE, (char *)sFileName)) != NULL)
        {
            if (!RpClumpStreamWrite(clump.GetClump(), stream))
            {
                char temp[256];
                sprintf(temp,"RenderWare %s Dff Exporter",VersionNumberText);
                MessageBox(hGMaxHWnd, "Failed to write clump to stream",
                    temp, MB_OK );
            }
            RwStreamClose(stream, NULL);

            ExportCustomData((char *)sFileName, m_topOfSelectedHierarchy);
        }
        else
        {
            char temp[256];
            sprintf(temp,"RenderWare %s Dff Exporter",VersionNumberText);
            MessageBox(hGMaxHWnd, "Failed to open stream",temp, MB_OK );
        }

        if (m_skinning)
        {

            RwFree(CSBoneIndexTags);
            CSBoneIndexTags = NULL;
            RwFree(skinFlags);
            skinFlags = NULL;
        }
    }
    while (ExportSelectNextObjectNode(gi)) ;

    /* Export data for this object.  Note the use of the original file-name and NULL parameter. */
    ExportCustomData((char *)filename, NULL);

    if (m_defaultAnimName)
    {
        RwFree(m_defaultAnimName);
    }

     /* Log size information for dff file. */
    m_WarningsList.writtenFile(filename);

    //export OK.
    return RwExpError("");
}

static void
AddNodePropertiesToFrame(INode *node, RwFrame *frame)
//Copy whatever contents user typed into user prop buffer (edit>properties...>user defined)
//for this node into a userdata plugin data attached to this frame
{
    TSTR buffer;
    TSTR linebuffer;
    RwInt32 numStrings = 0, stringNo = 0, userDataArrayIndex;
    RpUserDataArray *userDataArray;

    node->GetUserPropBuffer(buffer);

    while(buffer.length())
    {
        int newline = buffer.first('\n');

        if (newline == -1)
        {
            linebuffer = buffer;
            buffer.remove(0, buffer.length());
        }
        else
        {
            linebuffer = buffer;
            linebuffer.remove(newline-1, buffer.length()-newline+1);
            buffer.remove(0, newline+1);
        }

        numStrings++;
    }

    if (numStrings == 0)
    {
        return;
    }

    userDataArrayIndex = RwFrameAddUserDataArray(frame, "3dsmax User Properties", rpSTRINGUSERDATA, numStrings);
    userDataArray = RwFrameGetUserDataArray(frame, userDataArrayIndex);

    if (!userDataArray)
    {
        return;
    }

    node->GetUserPropBuffer(buffer);

    while(buffer.length())
    {
        int newline = buffer.first('\n');

        if (newline == -1)
        {
            linebuffer = buffer;
            buffer.remove(0, buffer.length());
        }
        else
        {
            linebuffer = buffer;
            linebuffer.remove(newline-1, buffer.length()-newline+1);
            buffer.remove(0, newline+1);
        }

        RpUserDataArraySetString(userDataArray, stringNo, linebuffer.data());
        stringNo++;
    }
}

static
bool HasNoEffectOnAnimation(
    //bone to check
    INode *boneNode,
    //all bones that move a vertex in any skin
    BonesContainer & influenceBones )
//We can try and reduce the number of frames and bones to only those we actually need
//Currently for Character Studio, which can add a lot of dummies to a character that don't do anything.
{
    //bone might effect animation if it has children
    if (boneNode->NumChildren() != 0)
    {
        return false;
    }

    //Character Studio footsteps are no good for anything so let's avoid them entirely
    if (IsAFootstepNode( boneNode ))
    {
        return true;
    }

    //if it's a biped part and it's hidden
    if (IsABipedNode( boneNode ) && boneNode->IsNodeHidden())
    {
        BonesContainer::iterator i = influenceBones.find( boneNode );

        if (i == influenceBones.end())
        {
            //it's a biped bone that doesn't influence any skins.
            return true;
        }
    }

    return false;
}

void
DFFExport::ConvertNode(INode* node, RpClump *clump, RwFrame *frame, RwBool lastChild, BonesContainer & influenceBones)
{
    RpAtomic *atomic = NULL;
    RwMatrix *matrix = NULL;
    Object *object;
    int nNumChildren, nChildNum;
    char *localNodeName;
    int localBoneIndex;
    int numBonesBeforeChildren;

    //NegativeScaleCheck( node, m_tvStart, m_WarningsList );

    localNodeName = node->GetName();

    //Object that is the result of the node's pipeline.
    //This is the object that the appears in the scene.
    //For example, if there is a Sphere in the scene that has a Bend and Taper applied,
    //EvalWorldState() would return an ObjectState containing a TriObject
    object = node->EvalWorldState(m_tvStart).obj;

    //make a matrix for the Renderware frame to match this node's transform
    matrix = RwFrameGetMatrix(frame);
    Matrix3 objMat = GetObjMatrix(node, m_tvStart, TRUE);

    //Character studio has a different idea of what is "up" than the rest of MAX.
    //Reorient this matrix to have the same "up" as everything else
    objMat = ToRenderwareAxisSystem( objMat, node );
    Matrix3ToRwMatrix(objMat, matrix);
    RwFrameUpdateObjects(frame);

    if (IsNodeRwParticle(node))
    {
        if (!ExportRwParticle(node,matrix,clump,frame,m_nScaleFactor,&m_WarningsList)) {
            m_WarningsList.add(CWarning(wtError,node->GetName(),
                "Error exporting renderware particle system"));
        }
    }

    //Increment the number of bones processed
    localBoneIndex = CSNumBones;
    CSNumBones++;
    //RpSkin is hardcoded with UInt8 indices to bones!
    if (m_skinning && CSNumBones > 256)
    {
        throw RwExpError("The selected hierarchy has too many bones!\n Renderware can only handle a maximum of 256 bones.");
    }

    //Grow the bone arrays to hold this one
    //TO DO: replace this with std::vector
    CSBoneIndexNodePointers = (INode **) RwRealloc(CSBoneIndexNodePointers, sizeof(INode *) * CSNumBones);
    CSBoneIndexTags = (RwInt32 *) RwRealloc(CSBoneIndexTags, sizeof(RwInt32) * CSNumBones);
    skinFlags = (RwUInt32 *)RwRealloc(skinFlags, sizeof(RwUInt32) * CSNumBones);
    skinInverseMatrices = (RwMatrix *)RwRealloc(skinInverseMatrices, sizeof(RwMatrix) * CSNumBones);

    {
        RWEXPMESSAGE(("%s(%d): %s = %d\n",
                      __FILE__, __LINE__,
                      node->GetName(), CSNumBones));
    }
    CSBoneIndexTags[localBoneIndex] = GetNodeTagValue(node);
    CSBoneIndexNodePointers[localBoneIndex] = node;
    skinFlags[localBoneIndex] = rpSKINANIMATEVERTEXGROUP | rpSKINPUSHPARENTMATRIX;
    RwMatrix *LTM = RwFrameGetLTM(frame);
    RwMatrixInvert(&skinInverseMatrices[localBoneIndex], LTM);

    // give customised exporter first refusal
    if (!ExportCustomNode(node, object, frame, clump))
    {
        //create the geometry
        //if the node has geometry in the Renderware RpGeometry sense of the word
        //and strictly is not a skin
        if (object && object->IsRenderable() && node->Renderable() && (!node->IsNodeHidden())
            && !IsSelSetSkinNode(node))
        {
            RwInt32 *vertexMap = NULL; RpAtomic *atomic = NULL; //return parameters not used here
            Matrix3 transform = m_objectOffset * m_scaleMatrix;

#ifdef RW_PATCH_EXP

            if (IsNodeRwPatch( node, m_tvStart))
            {
                ExtractGeometryFromPatchNode( node, clump, frame, atomic, transform, m_tvStart );
            }
            else

#endif /* RW_PATCH_EXP */

            {
                ExtractGeometryFromNode( node, clump, frame, atomic, vertexMap, transform );

                /* get rid of the vertex map */
                if (vertexMap)
                {
                    RwFree(vertexMap);
                }
            }


            SetCustomFrameData(node, frame); //DNGN
        }
    }

    if (m_anim || m_seqAnim)
    {
        _rpAnimSeqSetFrameTag(frame, CSBoneIndexTags[localBoneIndex]);
    }

    if (m_exportHAnim)
    {
        RpHAnimFrameSetID(frame, CSBoneIndexTags[localBoneIndex]);
    }

    if (m_exportLabels)
    {
        AddNodePropertiesToFrame(node, frame);
    }

    /* Always build the list of animation keyframes */
    AddSkinAnimToFrame(node, frame, m_defaultAnimName, localBoneIndex);

    if (m_anim)
    {
        AddAnimToFrame(node, frame, m_defaultAnimName);
    }

    numBonesBeforeChildren = CSNumBones;
    /* save the children */
    nNumChildren = node->NumberOfChildren();
    for (nChildNum = 0; nChildNum < nNumChildren; nChildNum++)
    {
        INode *childNode = node->GetChildNode(nChildNum);

        //Ignore is it doesn't affect the animation at all
        if (!HasNoEffectOnAnimation( childNode, influenceBones ))
        {
            //Allow a custom function to ignore hierarchy branches
            if (!CustomIgnoreNodeAndChildren(childNode))
            {
                //If the child is Physique or footsteps and has no children ignore it
                if (childNode->NumberOfChildren() > 0 ||
                    !(FindPhysiqueModifier(childNode) || IsAFootstepNode(childNode)))
                {

                    /* create the frame */
                    RwFrame *childFrame = RwFrameCreate();
                    if (childFrame)
                    {
                        RwFrameAddChild(frame, childFrame);
                        ConvertNode(childNode, clump, childFrame, nChildNum == (nNumChildren-1), influenceBones);

                        InitialiseCustomFrameData(m_topOfSelectedHierarchy, frame);
                    }
                }
            }
        }
    }

    RwFrameUpdateObjects(frame);
    RwFrameGetLTM(frame);


    if (numBonesBeforeChildren == CSNumBones)
    {
        /* I had no children on this bone so or in the POP flag */
        skinFlags[localBoneIndex] |= rpSKINPOPPARENTMATRIX;
    }
    else
    {
        /* Don't do the PUSH on my last child */
        skinFlags[lastChildIndex] &= ~rpSKINPUSHPARENTMATRIX;
    }

    lastChildIndex = localBoneIndex;
}

Box3
DFFExport::GetHierarchicalBbox(INode* npNode)
{
    Box3 ParentBbox;
    int numChildren = npNode->NumberOfChildren();

    /* inital values */
    ParentBbox.pmin.x = FLT_MAX;
    ParentBbox.pmin.y = FLT_MAX;
    ParentBbox.pmin.z = FLT_MAX;
    ParentBbox.pmax.x = -FLT_MAX;
    ParentBbox.pmax.y = -FLT_MAX;
    ParentBbox.pmax.z = -FLT_MAX;

    Object* pObj = npNode->EvalWorldState( m_tvStart ).obj;

    /* Only include renderable objects when working out the bounding-box */
    if (pObj && pObj->IsRenderable())
    {
        Matrix3 Mat = npNode->GetNodeTM( m_tvStart );
        /* take into account node offset */
        Mat.SetTrans(Mat.GetTrans() + npNode->GetObjOffsetPos());
        pObj->GetDeformBBox( m_tvStart, ParentBbox, &Mat );
    }

    /* for each child */
    for(int i=0; i<numChildren; i++)
    {
        /* get bbx */
        Box3 ChildBbox = GetHierarchicalBbox(npNode->GetChildNode(i));

        /* merge parent and child bounding boxes */
        if (ChildBbox.pmin.x < ParentBbox.pmin.x)
            ParentBbox.pmin.x = ChildBbox.pmin.x;
        if (ChildBbox.pmin.y < ParentBbox.pmin.y)
            ParentBbox.pmin.y = ChildBbox.pmin.y;
        if (ChildBbox.pmin.z < ParentBbox.pmin.z)
            ParentBbox.pmin.z = ChildBbox.pmin.z;

        if (ChildBbox.pmax.x > ParentBbox.pmax.x)
            ParentBbox.pmax.x = ChildBbox.pmax.x;
        if (ChildBbox.pmax.y > ParentBbox.pmax.y)
            ParentBbox.pmax.y = ChildBbox.pmax.y;
        if (ChildBbox.pmax.z > ParentBbox.pmax.z)
            ParentBbox.pmax.z = ChildBbox.pmax.z;
    }

    return ParentBbox;
}

float
DFFExport::ScaleTo_ScaleFactor(INode *node)
{
    Box3 boxBoundingBox = GetHierarchicalBbox(node);

    /* If m_nScaleType is 0, Scale To (1 is scale-by).
       Find the bounding-box of the object, and set the new scale-factor so that
       the object fits into a box of the size the user asked it to be scaled to.
       The scale-factor that will be applied to the vertices will be the value
       entered by the user divided by the size of the largest dimension of the bounding-box
       NB: This scale can depend on how the object is oriented. */

    float nScale;

    nScale = boxBoundingBox.pmax.x - boxBoundingBox.pmin.x;

    if (nScale < (boxBoundingBox.pmax.y - boxBoundingBox.pmin.y))
        nScale = boxBoundingBox.pmax.y - boxBoundingBox.pmin.y;

    if (nScale < (boxBoundingBox.pmax.z - boxBoundingBox.pmin.z))
        nScale = boxBoundingBox.pmax.z - boxBoundingBox.pmin.z;

    return m_nScaleFactor / nScale;
}

